<?php
// 1. প্রয়োজনীয় ফাইল ইনক্লুড
require_once 'db.php';
require_once 'config.php';

// 2. সেশন শুরু
session_start();

// 3. কারেন্ট ইউজার পাওয়া
$currentUser = getCurrentUser();
$userData = $currentUser ?: [];

// 4. AJAX রিকুয়েস্ট হ্যান্ডেলিং
if (isset($_GET['ajax'])) {
    header('Content-Type: application/json');
    
    switch ($_GET['ajax']) {
        case 'login':
            handleLogin();
            break;
        case 'register':
            handleRegister();
            break;
        case 'placeBet':
            handlePlaceBet();
            break;
        case 'submitDeposit':
            handleDeposit();
            break;
        case 'submitWithdraw':
            handleWithdraw();
            break;
        case 'getGameHistory':
            getGameHistory();
            break;
        case 'getMyBets':
            getMyBets();
            break;
        case 'getCommissionDetails':
            getCommissionDetails();
            break;
        case 'claimSalary':
            claimVipSalary();
            break;
        case 'redeemGift':
            redeemGiftCode();
            break;
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid request']);
    }
    exit;
}

// 5. লগআউট হ্যান্ডেলিং
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: index.php');
    exit;
}

// 6. পেজ নির্ধারণ
$page = isset($_GET['page']) ? $_GET['page'] : 'home';
if (!$currentUser && !in_array($page, ['login', 'register'])) {
    $page = 'login';
}

// ==============================================
// AJAX হ্যান্ডলার ফাংশন ডেফিনিশন
// ==============================================

function handleLogin() {
    global $pdo;
    
    $phone = $_POST['phone'] ?? '';
    $password = $_POST['password'] ?? '';
    
    if (empty($phone) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Please fill all fields']);
        return;
    }
    
    $user = getUserByPhone($phone);
    if (!$user) {
        echo json_encode(['success' => false, 'message' => 'User not found']);
        return;
    }
    
    if (!password_verify($password, $user['password'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid password']);
        return;
    }
    
    // সেশন তৈরি
    $_SESSION['user_uid'] = $user['uid'];
    $_SESSION['user_id'] = $user['id'];
    
    // লাস্ট লগিন আপডেট
    $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
    $stmt->execute([$user['id']]);
    
    echo json_encode(['success' => true, 'message' => 'Login successful']);
}

function handleRegister() {
    global $pdo;
    
    $phone = $_POST['phone'] ?? '';
    $password = $_POST['password'] ?? '';
    $refCode = $_POST['refCode'] ?? '';
    
    if (empty($phone) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Please fill all fields']);
        return;
    }
    
    if (!preg_match('/^[0-9]{10,11}$/', $phone)) {
        echo json_encode(['success' => false, 'message' => 'Invalid phone number']);
        return;
    }
    
    // ফোন নাম্বার চেক
    $existing = getUserByPhone($phone);
    if ($existing) {
        echo json_encode(['success' => false, 'message' => 'Phone already registered']);
        return;
    }
    
    // UIID জেনারেট
    $uiId = generateUiId();
    while (getUserByUiId($uiId)) {
        $uiId = generateUiId();
    }
    
    // রেফারার চেক
    $referrerId = null;
    if (!empty($refCode)) {
        $referrer = getUserByUiId($refCode);
        if ($referrer) {
            $referrerId = $referrer['id'];
        }
    }
    
    // ইউনিক UID
    $uid = uniqid('user_', true);
    
    try {
        // ইউজার তৈরি
        $stmt = $pdo->prepare("
            INSERT INTO users 
            (uid, name, phone, password, uiId, email, referrer, created_at) 
            VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $name = "User " . substr($uiId, -4);
        $email = $phone . "@hgzygame.com";
        $hashedPassword = password_hash($password, PASSWORD_BCRYPT);
        
        $stmt->execute([
            $uid,
            $name,
            $phone,
            $hashedPassword,
            $uiId,
            $email,
            $referrerId
        ]);
        
        $userId = $pdo->lastInsertId();
        
        // রেফারার আপডেট
        if ($referrerId) {
            $stmt = $pdo->prepare("
                UPDATE users 
                SET directSubs = directSubs + 1 
                WHERE id = ?
            ");
            $stmt->execute([$referrerId]);
        }
        
        echo json_encode(['success' => true, 'message' => 'Registration successful']);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Registration failed: ' . $e->getMessage()]);
    }
}

function handlePlaceBet() {
    global $pdo, $currentUser, $COLOR_MAP, $BET_MULTIPLIERS;
    
    if (!$currentUser) {
        echo json_encode(['success' => false, 'message' => 'Please login first']);
        return;
    }
    
    $gameType = $_POST['gameType'] ?? '';
    $period = $_POST['period'] ?? '';
    $betType = $_POST['betType'] ?? '';
    $amount = floatval($_POST['amount'] ?? 0);
    
    if ($amount < MIN_BET) {
        echo json_encode(['success' => false, 'message' => 'Minimum bet is ' . MIN_BET]);
        return;
    }
    
    // ব্যালেন্স চেক
    $totalBalance = $currentUser['depositBalance'] + $currentUser['winningBalance'];
    if ($totalBalance < $amount) {
        echo json_encode(['success' => false, 'message' => 'Insufficient balance']);
        return;
    }
    
    try {
        $pdo->beginTransaction();
        
        // ব্যালেন্স আপডেট
        if ($currentUser['depositBalance'] >= $amount) {
            $depositDeduct = $amount;
            $winningDeduct = 0;
        } else {
            $depositDeduct = $currentUser['depositBalance'];
            $winningDeduct = $amount - $depositDeduct;
        }
        
        $stmt = $pdo->prepare("
            UPDATE users 
            SET depositBalance = depositBalance - ?,
                winningBalance = winningBalance - ?,
                totalBetAmount = totalBetAmount + ?,
                turnoverReq = GREATEST(0, turnoverReq - ?)
            WHERE id = ?
        ");
        
        $stmt->execute([$depositDeduct, $winningDeduct, $amount, $amount, $currentUser['id']]);
        
        // বেট সেভ
        $potentialWin = $amount * ($BET_MULTIPLIERS[$betType] ?? 2);
        
        $stmt = $pdo->prepare("
            INSERT INTO bets 
            (user_id, uid, game_type, period, bet_type, amount, potential_win, outcome, created_at)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $currentUser['id'],
            $currentUser['uid'],
            $gameType,
            $period,
            $betType,
            $amount,
            $potentialWin
        ]);
        
        $betId = $pdo->lastInsertId();
        
        // ট্রানজেকশন লগ
        logTransaction(
            $currentUser['id'],
            'bet',
            -$amount,
            $betId,
            "Bet on {$gameType} - {$betType}"
        );
        
        // কমিশন (0.6%)
        if ($currentUser['referrer']) {
            $commission = $amount * COMM_RATE;
            $referrer = getUserById($currentUser['referrer']);
            
            if ($referrer) {
                // রেফারার ব্যালেন্স আপডেট
                $stmt = $pdo->prepare("
                    UPDATE users 
                    SET winningBalance = winningBalance + ?,
                        totalCommission = totalCommission + ?
                    WHERE id = ?
                ");
                $stmt->execute([$commission, $commission, $referrer['id']]);
                
                // কমিশন লগ
                $stmt = $pdo->prepare("
                    INSERT INTO commissions 
                    (user_id, from_user_id, type, amount, description, bet_id, created_at)
                    VALUES (?, ?, 'Bet Comm', ?, ?, ?, NOW())
                ");
                $stmt->execute([
                    $referrer['id'],
                    $currentUser['id'],
                    $commission,
                    "Commission from {$currentUser['name']}'s bet",
                    $betId
                ]);
                
                // ট্রানজেকশন লগ
                logTransaction(
                    $referrer['id'],
                    'commission',
                    $commission,
                    $betId,
                    "Commission from {$currentUser['name']}"
                );
            }
        }
        
        $pdo->commit();
        
        // ইউজার ডাটা রিফ্রেশ
        $currentUser = getUserById($currentUser['id']);
        $_SESSION['current_user'] = $currentUser;
        
        echo json_encode(['success' => true, 'message' => 'Bet placed successfully']);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Bet failed: ' . $e->getMessage()]);
    }
}

function handleDeposit() {
    global $pdo, $currentUser;
    
    if (!$currentUser) {
        echo json_encode(['success' => false, 'message' => 'Please login first']);
        return;
    }
    
    $amount = floatval($_POST['amount'] ?? 0);
    $method = $_POST['method'] ?? '';
    $transactionId = $_POST['transactionId'] ?? '';
    
    if ($amount < MIN_DEPOSIT) {
        echo json_encode(['success' => false, 'message' => 'Minimum deposit is ' . MIN_DEPOSIT]);
        return;
    }
    
    if (empty($transactionId)) {
        echo json_encode(['success' => false, 'message' => 'Please enter Transaction ID']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO deposits 
            (user_id, uid, amount, method, transactionId, status, created_at)
            VALUES (?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $currentUser['id'],
            $currentUser['uid'],
            $amount,
            $method,
            $transactionId
        ]);
        
        // ট্রানজেকশন লগ
        logTransaction(
            $currentUser['id'],
            'deposit',
            0,
            $transactionId,
            "Deposit request - ৳{$amount} via {$method}"
        );
        
        echo json_encode(['success' => true, 'message' => 'Deposit request submitted']);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Deposit failed: ' . $e->getMessage()]);
    }
}

function handleWithdraw() {
    global $pdo, $currentUser;
    
    if (!$currentUser) {
        echo json_encode(['success' => false, 'message' => 'Please login first']);
        return;
    }
    
    $amount = floatval($_POST['amount'] ?? 0);
    $method = $_POST['method'] ?? '';
    $walletNumber = $_POST['walletNumber'] ?? '';
    
    if ($amount < MIN_WITHDRAW) {
        echo json_encode(['success' => false, 'message' => 'Minimum withdrawal is ' . MIN_WITHDRAW]);
        return;
    }
    
    if ($amount > MAX_WITHDRAW) {
        echo json_encode(['success' => false, 'message' => 'Maximum withdrawal is ' . MAX_WITHDRAW]);
        return;
    }
    
    if ($currentUser['winningBalance'] < $amount) {
        echo json_encode(['success' => false, 'message' => 'Insufficient winning balance']);
        return;
    }
    
    if ($currentUser['turnoverReq'] > 0) {
        echo json_encode(['success' => false, 'message' => 'Turnover requirement not met']);
        return;
    }
    
    if (!preg_match('/^01[3-9]\d{8}$/', $walletNumber)) {
        echo json_encode(['success' => false, 'message' => 'Invalid wallet number']);
        return;
    }
    
    try {
        // উইথড্র রিকুয়েস্ট
        $stmt = $pdo->prepare("
            INSERT INTO withdrawals 
            (user_id, uid, amount, method, wallet_number, status, created_at)
            VALUES (?, ?, ?, ?, ?, 'pending', NOW())
        ");
        
        $stmt->execute([
            $currentUser['id'],
            $currentUser['uid'],
            $amount,
            $method,
            $walletNumber
        ]);
        
        // ট্রানজেকশন লগ
        logTransaction(
            $currentUser['id'],
            'withdrawal',
            0,
            '',
            "Withdrawal request - ৳{$amount} to {$walletNumber}"
        );
        
        echo json_encode(['success' => true, 'message' => 'Withdrawal request submitted']);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Withdrawal failed: ' . $e->getMessage()]);
    }
}

function getGameHistory() {
    global $pdo;
    
    $gameType = $_GET['game'] ?? '30s';
    $limit = 10;
    
    // সিমুলেটেড হিস্ট্রি (আসল ইমপ্লিমেন্টেশনে ডাটাবেস থেকে নিতে হবে)
    $history = [];
    $now = time();
    
    for ($i = 1; $i <= $limit; $i++) {
        $time = $now - ($i * 30); // 30 সেকেন্ড ইন্টারভাল
        $number = rand(0, 9);
        
        $history[] = [
            'period' => date('Ymd', $time) . str_pad($i, 4, '0', STR_PAD_LEFT),
            'number' => $number,
            'color' => getColorForNumber($number),
            'size' => $number >= 5 ? 'Big' : 'Small'
        ];
    }
    
    echo json_encode(['success' => true, 'history' => $history]);
}

function getMyBets() {
    global $pdo, $currentUser;
    
    if (!$currentUser) {
        echo json_encode(['success' => false, 'message' => 'Please login first']);
        return;
    }
    
    $gameType = $_GET['game'] ?? '30s';
    $limit = 10;
    
    $stmt = $pdo->prepare("
        SELECT * FROM bets 
        WHERE user_id = ? AND game_type = ?
        ORDER BY created_at DESC 
        LIMIT ?
    ");
    
    $stmt->execute([$currentUser['id'], $gameType, $limit]);
    $bets = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'bets' => $bets]);
}

function getCommissionDetails() {
    global $pdo, $currentUser;
    
    if (!$currentUser) {
        echo json_encode(['success' => false, 'message' => 'Please login first']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT c.*, u.uiId as from_uiId 
        FROM commissions c
        LEFT JOIN users u ON c.from_user_id = u.id
        WHERE c.user_id = ?
        ORDER BY c.created_at DESC 
        LIMIT 20
    ");
    
    $stmt->execute([$currentUser['id']]);
    $commissions = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'commissions' => $commissions]);
}

function claimVipSalary() {
    global $pdo, $currentUser, $VIP_LEVELS;
    
    if (!$currentUser) {
        echo json_encode(['success' => false, 'message' => 'Please login first']);
        return;
    }
    
    // VIP লেভেল বের করা
    $vipLevel = 0;
    foreach (array_reverse($VIP_LEVELS) as $level) {
        if ($currentUser['totalBetAmount'] >= $level['min']) {
            $vipLevel = $level['level'];
            $salary = $level['salary'];
            break;
        }
    }
    
    if ($salary <= 0) {
        echo json_encode(['success' => false, 'message' => 'No salary available for your VIP level']);
        return;
    }
    
    // চেক করা হয়েছে কিনা (এই মাসে)
    $thisMonth = date('Y-m-01');
    
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as claimed 
        FROM transactions 
        WHERE user_id = ? 
        AND type = 'bonus' 
        AND description LIKE '%VIP Salary%' 
        AND created_at >= ?
    ");
    
    $stmt->execute([$currentUser['id'], $thisMonth]);
    $result = $stmt->fetch();
    
    if ($result['claimed'] > 0) {
        echo json_encode(['success' => false, 'message' => 'Salary already claimed this month']);
        return;
    }
    
    try {
        $pdo->beginTransaction();
        
        // ব্যালেন্স আপডেট
        $stmt = $pdo->prepare("
            UPDATE users 
            SET winningBalance = winningBalance + ?
            WHERE id = ?
        ");
        
        $stmt->execute([$salary, $currentUser['id']]);
        
        // ট্রানজেকশন লগ
        logTransaction(
            $currentUser['id'],
            'bonus',
            $salary,
            null,
            "VIP {$vipLevel} Monthly Salary"
        );
        
        $pdo->commit();
        
        echo json_encode(['success' => true, 'amount' => $salary, 'message' => 'Salary claimed']);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to claim salary']);
    }
}

function redeemGiftCode() {
    global $pdo, $currentUser;
    
    if (!$currentUser) {
        echo json_encode(['success' => false, 'message' => 'Please login first']);
        return;
    }
    
    $code = $_POST['code'] ?? '';
    
    if (empty($code)) {
        echo json_encode(['success' => false, 'message' => 'Please enter gift code']);
        return;
    }
    
    // সিম্পল গিফট কোড চেক (আসল ইমপ্লিমেন্টেশনে গিফট কোড টেবিল থাকবে)
    $validCodes = [
        'WELCOME100' => 100,
        'HGZY2024' => 50,
        'BONUS500' => 500
    ];
    
    if (!isset($validCodes[$code])) {
        echo json_encode(['success' => false, 'message' => 'Invalid gift code']);
        return;
    }
    
    $amount = $validCodes[$code];
    
    // চেক করা হয়েছে কিনা
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as used 
        FROM transactions 
        WHERE user_id = ? 
        AND type = 'bonus' 
        AND description LIKE ?
    ");
    
    $stmt->execute([$currentUser['id'], "%Gift code: {$code}%"]);
    $result = $stmt->fetch();
    
    if ($result['used'] > 0) {
        echo json_encode(['success' => false, 'message' => 'Gift code already used']);
        return;
    }
    
    try {
        $pdo->beginTransaction();
        
        // ব্যালেন্স আপডেট
        $stmt = $pdo->prepare("
            UPDATE users 
            SET winningBalance = winningBalance + ?
            WHERE id = ?
        ");
        
        $stmt->execute([$amount, $currentUser['id']]);
        
        // ট্রানজেকশন লগ
        logTransaction(
            $currentUser['id'],
            'bonus',
            $amount,
            $code,
            "Gift code: {$code}"
        );
        
        $pdo->commit();
        
        echo json_encode(['success' => true, 'amount' => $amount, 'message' => 'Gift redeemed']);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Failed to redeem gift code']);
    }
}

// হেল্পার ফাংশন
function getColorForNumber($number) {
    global $COLOR_MAP;
    return isset($COLOR_MAP[$number]) ? implode('/', $COLOR_MAP[$number]) : '';
}

// ==============================================
// HTML শুরু
// ==============================================
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <title><?php echo SITE_NAME; ?> - Ultimate</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdnjs.cloudflare.com/ajax/libs/qrcodejs/1.0.0/qrcode.min.js"></script>
</head>
<body>
    <div class="container">
        <?php if (!$currentUser): ?>
        <!-- ================= AUTH PAGE ================= -->
        <div id="auth-page" class="auth-wrap">
            <div class="auth-nav-top">
                <i class="fas fa-chevron-left hidden" id="auth-back-btn" onclick="switchAuth('login')"></i>
                <span class="auth-title"><?php echo SITE_NAME; ?></span>
                <div class="lang-flag"><img src="https://flagcdn.com/w40/us.png" alt="EN" style="width:20px"> EN</div>
            </div>

            <!-- Login Section -->
            <div id="login-section">
                <div class="auth-header-text">
                    <h2>Login</h2>
                    <p>Please enter your phone number</p>
                </div>
                <form id="login-form" onsubmit="return submitLogin(event)">
                    <div class="inp-field">
                        <i class="fas fa-mobile-alt icon"></i>
                        <span class="prefix">+880</span>
                        <input type="tel" id="login-phone" placeholder="Please enter the phone number" required>
                    </div>
                    <div class="inp-field">
                        <i class="fas fa-lock icon"></i>
                        <input type="password" id="login-password" placeholder="Please enter password" required>
                        <i class="fas fa-eye-slash eye" onclick="togglePass('login-password')"></i>
                    </div>
                    <button type="submit" class="auth-main-btn">Log In</button>
                    <button type="button" class="auth-sec-btn" onclick="switchAuth('register')">Register Account</button>
                </form>
            </div>

            <!-- Register Section -->
            <div id="register-section" class="hidden">
                <div class="auth-header-text">
                    <h2>Register</h2>
                    <p>Create a new account</p>
                </div>
                <form id="register-form" onsubmit="return submitRegister(event)">
                    <div class="inp-field">
                        <i class="fas fa-mobile-alt icon"></i>
                        <span class="prefix">+880</span>
                        <input type="tel" id="reg-phone" placeholder="Please enter the phone number" required>
                    </div>
                    <div class="inp-field">
                        <i class="fas fa-lock icon"></i>
                        <input type="password" id="reg-pass" placeholder="Set password" required>
                    </div>
                    <div class="inp-field">
                        <i class="fas fa-lock icon"></i>
                        <input type="password" id="reg-pass-conf" placeholder="Confirm password" required>
                    </div>
                    <div class="inp-field">
                        <i class="fas fa-gift icon"></i>
                        <input type="text" id="reg-ref" placeholder="Invitation code" value="<?php echo isset($_GET['invitationCode']) ? $_GET['invitationCode'] : ''; ?>">
                    </div>
                    <div class="agree-box">
                        <input type="checkbox" id="reg-agree" required>
                        <label for="reg-agree">I have read and agree to the <span class="text-red">Privacy Agreement</span></label>
                    </div>
                    <button type="submit" class="auth-main-btn">Register</button>
                    <button type="button" class="auth-sec-btn" onclick="switchAuth('login')">Have an account? Login</button>
                </form>
            </div>
        </div>
        <?php else: ?>
        <!-- ================= MAIN APP ================= -->
        <div id="main-app">
            <!-- HEADER -->
            <header class="header">
                <div class="header-left">
                    <img src="ball_0.webp" class="app-logo" alt="Logo">
                    <span class="app-name"><?php echo SITE_NAME; ?></span>
                </div>
                <div class="balance-chip" onclick="showPage('wallet')">
                    <i class="fas fa-wallet"></i> ৳<span id="header-balance"><?php echo number_format($userData['depositBalance'] + $userData['winningBalance'], 2); ?></span>
                    <i class="fas fa-plus-circle add-btn"></i>
                </div>
            </header>

            <!-- MARQUEE NOTICE -->
            <div class="notice-bar">
                <i class="fas fa-volume-up"></i>
                <marquee scrollamount="5" id="notice-text">
                    Welcome! Get Bonus on 1st deposit. Earn <?php echo (getCommissionRate() * 100); ?>% commission on every bet from your team. Play responsibly.
                </marquee>
            </div>

            <!-- DYNAMIC PAGE CONTENT -->
            <div id="page-content">
                <?php 
                $pageFile = 'pages/' . $page . '.php';
                if (file_exists($pageFile)) {
                    include $pageFile;
                } else {
                    include 'pages/home.php';
                }
                ?>
            </div>

            <!-- BOTTOM NAVIGATION -->
            <div class="bottom-nav">
                <button class="nav-item <?php echo $page == 'home' ? 'active' : ''; ?>" onclick="showPage('home')">
                    <i class="fas fa-home"></i> <span>Home</span>
                </button>
                <button class="nav-item <?php echo $page == 'activity' ? 'active' : ''; ?>" onclick="showPage('activity')">
                    <i class="fas fa-fire"></i> <span>Activity</span>
                </button>
                <button class="nav-item <?php echo $page == 'promotion' ? 'active' : ''; ?>" onclick="showPage('promotion')">
                    <i class="fas fa-diamond"></i> <span>Promotion</span>
                </button>
                <button class="nav-item <?php echo $page == 'wallet' ? 'active' : ''; ?>" onclick="showPage('wallet')">
                    <i class="fas fa-wallet"></i> <span>Wallet</span>
                </button>
                <button class="nav-item <?php echo $page == 'profile' ? 'active' : ''; ?>" onclick="showPage('profile')">
                    <i class="fas fa-user"></i> <span>Profile</span>
                </button>
            </div>
        </div>
        <?php endif; ?>
    </div>

    <!-- POPUPS -->
    <div id="bet-overlay" class="overlay">
        <div class="popup-bottom">
            <div class="popup-header">
                <h3 id="bet-title">Bet</h3>
                <i class="fas fa-times" onclick="closePopup()"></i>
            </div>
            <div class="popup-body">
                <p>Balance: <span class="text-gold">৳<span id="bet-bal"><?php echo number_format($userData['depositBalance'] + $userData['winningBalance'], 2); ?></span></span></p>
                <div class="bet-input-row">
                    <span>৳</span>
                    <input type="number" id="bet-amount-in" value="10" min="10">
                </div>
                <div class="chip-row">
                    <button type="button" onclick="addBet(10)">+10</button>
                    <button type="button" onclick="addBet(50)">+50</button>
                    <button type="button" onclick="addBet(100)">+100</button>
                    <button type="button" onclick="addBet(1000)">+1K</button>
                    <button type="button" onclick="mulBet(2)" class="btn-mul">x2</button>
                    <button type="button" onclick="mulBet(3)" class="btn-mul">x3</button>
                </div>
                <div class="total-bet-info">Total Contract: <span id="total-contract">10</span></div>
                <button class="full-btn btn-primary" onclick="confirmBet()">Confirm Bet</button>
            </div>
        </div>
    </div>

    <div id="res-overlay" class="overlay">
        <div class="popup-center win-popup">
            <div class="win-header"><img src="ball_0.webp" class="win-logo"></div>
            <h2>Congratulations!</h2>
            <p>You Won</p>
            <div class="win-amount" id="res-amt">+৳0.00</div>
            <button class="full-btn btn-primary" onclick="closeResPopup()">Awesome</button>
        </div>
    </div>

    <div id="qr-overlay" class="overlay">
        <div class="popup-center qr-box">
            <h3>Invite Friends</h3>
            <div id="qrcode" style="margin:20px auto; display:flex; justify-content:center;"></div>
            <p style="font-size:12px; color:#888; margin-bottom:10px;">Scan or Copy Link</p>
            <div class="link-copy-box">
                <input type="text" id="share-link-inp" readonly>
                <button onclick="copyShareLink()">Copy</button>
            </div>
            <button class="full-btn btn-primary" onclick="document.getElementById('qr-overlay').style.display='none'">Close</button>
        </div>
    </div>

    <div id="alert-overlay" class="overlay">
        <div class="popup-center alert-box">
            <h3 id="alert-t">Alert</h3>
            <p id="alert-m">Message here</p>
            <button class="full-btn btn-primary" onclick="document.getElementById('alert-overlay').style.display='none'">OK</button>
        </div>
    </div>

    <script>
        // গ্লোবাল ভেরিয়েবল
        let currentUser = <?php echo json_encode($currentUser ? $userData : null); ?>;
        let activeGame = null;
        let timerInt = null;
        let activeBet = null;
        let activeBetType = '';
        let curPeriod = null;
        let depositMethod = 'bKash';
        
        const COMM_RATE = <?php echo getCommissionRate(); ?>;
        const MIN_BET = <?php echo MIN_BET; ?>;
        const MIN_DEPOSIT = <?php echo MIN_DEPOSIT; ?>;
        const MIN_WITHDRAW = <?php echo MIN_WITHDRAW; ?>;
        const MAX_WITHDRAW = <?php echo MAX_WITHDRAW; ?>;
        
        const GAMES = {
            '30s': { name: 'Win Go 30s', time: 30, db: 'wingo30s' },
            '1m': { name: 'Win Go 1m', time: 60, db: 'wingo1m' },
            '3m': { name: 'Win Go 3m', time: 180, db: 'wingo3m' },
            '5m': { name: 'Win Go 5m', time: 300, db: 'wingo5m' },
            '10m': { name: 'Win Go 10m', time: 600, db: 'wingo10m' },
            '20m': { name: 'Win Go 20m', time: 1200, db: 'wingo20m' }
        };
        
        const VIP_LEVELS = <?php echo json_encode($VIP_LEVELS); ?>;
        
        // বেট মাল্টিপ্লায়ার
        const BET_MULTIPLIERS = {
            'Green': 2, 'Red': 2, 'Violet': 4.5,
            'Big': 2, 'Small': 2,
            '0': 9, '1': 9, '2': 9, '3': 9, '4': 9,
            '5': 9, '6': 9, '7': 9, '8': 9, '9': 9
        };
        
        // কালার ম্যাপ
        const COLOR_MAP = {
            0: ['Red', 'Violet'],
            1: ['Green'],
            2: ['Red'],
            3: ['Green'],
            4: ['Red'],
            5: ['Green', 'Violet'],
            6: ['Red'],
            7: ['Green'],
            8: ['Red'],
            9: ['Green']
        };
        
        // পেজ নেভিগেশন
        function showPage(page) {
            window.location.href = 'index.php?page=' + page;
        }
        
        function goHome() {
            showPage('home');
        }
        
        // অথেন্টিকেশন ফাংশন
        function switchAuth(type) {
            document.getElementById('login-section').classList.toggle('hidden', type !== 'login');
            document.getElementById('register-section').classList.toggle('hidden', type !== 'register');
            document.getElementById('auth-back-btn').classList.toggle('hidden', type === 'login');
        }
        
        function togglePass(id) {
            const x = document.getElementById(id);
            x.type = x.type === "password" ? "text" : "password";
        }
        
        async function submitLogin(e) {
            e.preventDefault();
            const phone = document.getElementById('login-phone').value;
            const password = document.getElementById('login-password').value;
            
            if (!phone || !password) {
                alertMsg('Error', 'Please fill all fields');
                return false;
            }
            
            try {
                const response = await fetch('index.php?ajax=login', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `phone=${encodeURIComponent(phone)}&password=${encodeURIComponent(password)}`
                });
                
                const result = await response.json();
                if (result.success) {
                    window.location.reload();
                } else {
                    alertMsg('Error', result.message);
                }
            } catch (error) {
                alertMsg('Error', 'Network error. Please try again.');
            }
        }
        
        async function submitRegister(e) {
            e.preventDefault();
            const phone = document.getElementById('reg-phone').value;
            const password = document.getElementById('reg-pass').value;
            const confirmPass = document.getElementById('reg-pass-conf').value;
            const refCode = document.getElementById('reg-ref').value;
            const agree = document.getElementById('reg-agree').checked;
            
            if (password !== confirmPass) {
                alertMsg('Error', 'Passwords do not match!');
                return false;
            }
            
            if (!agree) {
                alertMsg('Error', 'Please agree to Privacy Policy');
                return false;
            }
            
            if (!phone.match(/^[0-9]{10,11}$/)) {
                alertMsg('Error', 'Invalid phone number');
                return false;
            }
            
            try {
                const response = await fetch('index.php?ajax=register', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `phone=${encodeURIComponent(phone)}&password=${encodeURIComponent(password)}&refCode=${encodeURIComponent(refCode)}`
                });
                
                const result = await response.json();
                if (result.success) {
                    alertMsg('Success', 'Account Created!');
                    setTimeout(() => {
                        switchAuth('login');
                    }, 1500);
                } else {
                    alertMsg('Error', result.message);
                }
            } catch (error) {
                alertMsg('Error', 'Network error. Please try again.');
            }
        }
        
        // গেম ফাংশন
        function openGame(type) {
            activeGame = type;
            showPage('game&type=' + type);
        }
        
        function runTimer() {
            if (!activeGame) return;
            if (timerInt) clearInterval(timerInt);
            
            const dur = GAMES[activeGame].time;
            timerInt = setInterval(() => {
                const now = new Date();
                const totalSec = Math.floor(now.getTime() / 1000);
                const left = dur - (totalSec % dur);
                const m = Math.floor(left / 60);
                const s = left % 60;
                
                if (document.getElementById('countdown')) {
                    document.getElementById('countdown').innerText = `${m<10?'0'+m:m}:${s<10?'0'+s:s}`;
                }
                
                const dateStr = now.toISOString().slice(0, 10).replace(/-/g, '');
                const dayIndex = Math.floor(((now.getHours()*3600)+(now.getMinutes()*60)+now.getSeconds()) / dur) + 1;
                curPeriod = dateStr + String(100000 + dayIndex).substring(1);
                
                if (document.getElementById('period-num')) {
                    document.getElementById('period-num').innerText = curPeriod;
                }
                
                const gameFrame = document.querySelector('.game-frame');
                if (gameFrame) {
                    if (left <= 5) {
                        gameFrame.style.opacity = 0.8;
                    } else {
                        gameFrame.style.opacity = 1;
                    }
                }
                
                if (left <= 1) {
                    setTimeout(() => {
                        loadGameHistory();
                    }, 2000);
                }
            }, 1000);
        }
        
        async function loadGameHistory() {
            if (!activeGame) return;
            
            try {
                const response = await fetch(`index.php?ajax=getGameHistory&game=${activeGame}`);
                const result = await response.json();
                
                if (result.success && document.getElementById('history-body')) {
                    renderHistory(result.history);
                }
            } catch (error) {
                console.error('Error loading history:', error);
            }
        }
        
        function renderHistory(list) {
            let html = '';
            list.forEach(i => {
                const n = parseInt(i.number);
                const sz = n >= 5 ? 'Big' : 'Small';
                let txtClass = 'text-c-red';
                
                if ([1,3,7,9].includes(n)) txtClass = 'text-c-green';
                else if (n === 0) txtClass = 'text-grad-0';
                else if (n === 5) txtClass = 'text-grad-5';
                
                let dotHtml = '';
                if (n === 0) {
                    dotHtml = `<div class="dot-wrap"><span class="dot d-red"></span><span class="dot d-violet"></span></div>`;
                } else if (n === 5) {
                    dotHtml = `<div class="dot-wrap"><span class="dot d-green"></span><span class="dot d-violet"></span></div>`;
                } else {
                    const dCol = [1,3,7,9].includes(n) ? 'd-green' : 'd-red';
                    dotHtml = `<div class="dot-wrap"><span class="dot ${dCol}"></span></div>`;
                }
                
                html += `<tr>
                    <td>${i.period ? i.period.slice(-4) : ''}</td>
                    <td class="res-num-text ${txtClass}">${n}</td>
                    <td>${sz}</td>
                    <td>${dotHtml}</td>
                </tr>`;
            });
            
            if (document.getElementById('history-body')) {
                document.getElementById('history-body').innerHTML = html;
            }
        }
        
        // হিস্ট্রি ট্যাব সুইচ
        function switchHistory(type) {
            const tabs = document.querySelectorAll('.tab-item');
            tabs.forEach(tab => tab.classList.remove('active'));
            
            if (type === 'game') {
                tabs[0].classList.add('active');
                loadGameHistory();
            } else {
                tabs[1].classList.add('active');
                loadMyBets();
            }
        }
        
        async function loadMyBets() {
            if (!activeGame) return;
            
            try {
                const response = await fetch(`index.php?ajax=getMyBets&game=${activeGame}`);
                const result = await response.json();
                
                if (result.success && document.getElementById('history-body')) {
                    let html = '';
                    result.bets.forEach(bet => {
                        html += `<tr>
                            <td>${bet.period.slice(-4)}</td>
                            <td>${bet.bet_type}</td>
                            <td>${bet.amount}</td>
                            <td>${bet.outcome}</td>
                        </tr>`;
                    });
                    document.getElementById('history-body').innerHTML = html;
                }
            } catch (error) {
                console.error('Error loading my bets:', error);
            }
        }
        
        // বেটিং ফাংশন
        function openBetPopup(type) {
            activeBetType = type;
            document.getElementById('bet-title').innerText = "Bet " + type;
            document.getElementById('bet-overlay').style.display = 'flex';
        }
        
        function closePopup() {
            document.getElementById('bet-overlay').style.display = 'none';
        }
        
        function addBet(n) {
            const input = document.getElementById('bet-amount-in');
            input.value = parseInt(input.value || 0) + n;
            updateContract();
        }
        
        function mulBet(n) {
            const input = document.getElementById('bet-amount-in');
            input.value = parseInt(input.value || 0) * n;
            updateContract();
        }
        
        function updateContract() {
            const input = document.getElementById('bet-amount-in');
            document.getElementById('total-contract').innerText = input.value || 0;
        }
        
        async function confirmBet() {
            const amt = parseInt(document.getElementById('bet-amount-in').value);
            if (amt < MIN_BET) {
                alertMsg("Error", "Min bet " + MIN_BET);
                return;
            }
            
            if (!currentUser) {
                alertMsg("Error", "Please login first");
                return;
            }
            
            const total = parseFloat(currentUser.depositBalance) + parseFloat(currentUser.winningBalance);
            if (total < amt) {
                alertMsg("Error", "Low Balance");
                return;
            }
            
            if (!activeGame || !curPeriod) {
                alertMsg("Error", "Game not loaded");
                return;
            }
            
            try {
                const response = await fetch('index.php?ajax=placeBet', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `gameType=${activeGame}&period=${curPeriod}&betType=${activeBetType}&amount=${amt}`
                });
                
                const result = await response.json();
                if (result.success) {
                    closePopup();
                    alertMsg("Success", "Bet Placed");
                    
                    // ব্যালেন্স আপডেট
                    if (currentUser) {
                        if (currentUser.depositBalance >= amt) {
                            currentUser.depositBalance -= amt;
                        } else {
                            currentUser.winningBalance -= (amt - currentUser.depositBalance);
                            currentUser.depositBalance = 0;
                        }
                        updateUI();
                    }
                } else {
                    alertMsg("Error", result.message);
                }
            } catch (error) {
                alertMsg("Error", "Network error");
            }
        }
        
        // ওয়ালেট ফাংশন
        function selDep(m) {
            depositMethod = m;
            document.querySelectorAll('.pay-card').forEach(i => i.classList.remove('active'));
            document.getElementById('pay-' + m.toLowerCase()).classList.add('active');
            loadAgentNum();
        }
        
        function loadAgentNum() {
            const nums = {
                'bKash': '<?php echo getAgentNumber("bkash"); ?>',
                'Nagad': '<?php echo getAgentNumber("nagad"); ?>',
                'Rocket': '<?php echo getAgentNumber("rocket"); ?>'
            };
            
            if (document.getElementById('agent-num')) {
                document.getElementById('agent-num').innerText = nums[depositMethod];
            }
        }
        
        function copyAgent() {
            const num = document.getElementById('agent-num').innerText;
            navigator.clipboard.writeText(num);
            alertMsg("Copied", "Number Copied");
        }
        
        async function submitDeposit() {
            const amt = parseInt(document.getElementById('dep-amount').value);
            const trxId = document.getElementById('dep-trx').value;
            
            if (amt < MIN_DEPOSIT) {
                alertMsg("Error", "Min Deposit " + MIN_DEPOSIT);
                return;
            }
            
            if (!trxId.trim()) {
                alertMsg("Error", "Please enter Transaction ID");
                return;
            }
            
            try {
                const response = await fetch('index.php?ajax=submitDeposit', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `amount=${amt}&method=${depositMethod}&transactionId=${encodeURIComponent(trxId)}`
                });
                
                const result = await response.json();
                if (result.success) {
                    alertMsg("Submitted", "Wait for Admin Approval.");
                    showPage('wallet');
                } else {
                    alertMsg("Error", result.message);
                }
            } catch (error) {
                alertMsg("Error", "Network error");
            }
        }
        
        async function submitWithdraw() {
            const method = document.getElementById('w-method').value;
            const number = document.getElementById('w-number').value;
            const amount = parseInt(document.getElementById('w-amount').value);
            
            if (amount < MIN_WITHDRAW) {
                alertMsg("Error", "Min Withdrawal " + MIN_WITHDRAW);
                return;
            }
            
            if (amount > MAX_WITHDRAW) {
                alertMsg("Error", "Max Withdrawal " + MAX_WITHDRAW);
                return;
            }
            
            if (!currentUser || parseFloat(currentUser.winningBalance) < amount) {
                alertMsg("Error", "Insufficient winning balance");
                return;
            }
            
            if (parseFloat(currentUser.turnoverReq) > 0) {
                alertMsg("Error", "Turnover not met!");
                return;
            }
            
            if (!number.match(/^01[3-9]\d{8}$/)) {
                alertMsg("Error", "Invalid wallet number");
                return;
            }
            
            try {
                const response = await fetch('index.php?ajax=submitWithdraw', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `amount=${amount}&method=${method}&walletNumber=${encodeURIComponent(number)}`
                });
                
                const result = await response.json();
                if (result.success) {
                    alertMsg("Success", "Withdraw Requested");
                    showPage('wallet');
                } else {
                    alertMsg("Error", result.message);
                }
            } catch (error) {
                alertMsg("Error", "Network error");
            }
        }
        
        // VIP ফাংশন
        async function claimSalary() {
            try {
                const response = await fetch('index.php?ajax=claimSalary');
                const result = await response.json();
                
                if (result.success) {
                    alertMsg("Success", "Salary claimed successfully!");
                    if (currentUser) {
                        currentUser.winningBalance = parseFloat(currentUser.winningBalance) + result.amount;
                        updateUI();
                    }
                } else {
                    alertMsg("Error", result.message);
                }
            } catch (error) {
                alertMsg("Error", "Network error");
            }
        }
        
        // গিফট কোড
        async function submitGift() {
            const code = document.getElementById('gift-code-in').value;
            if (!code.trim()) {
                alertMsg("Error", "Please enter gift code");
                return;
            }
            
            try {
                const response = await fetch('index.php?ajax=redeemGift', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `code=${encodeURIComponent(code)}`
                });
                
                const result = await response.json();
                if (result.success) {
                    alertMsg("Success", "Gift redeemed successfully!");
                    if (currentUser) {
                        currentUser.winningBalance = parseFloat(currentUser.winningBalance) + result.amount;
                        updateUI();
                    }
                } else {
                    alertMsg("Error", result.message);
                }
            } catch (error) {
                alertMsg("Error", "Network error");
            }
        }
        
        // ইনভাইট ফাংশন
        function openInviteModal() {
            if (!currentUser) return;
            
            document.getElementById('qr-overlay').style.display = 'flex';
            const link = window.location.origin + '/index.php?invitationCode=' + currentUser.uiId;
            document.getElementById('share-link-inp').value = link;
            
            const qrcodeDiv = document.getElementById('qrcode');
            qrcodeDiv.innerHTML = "";
            new QRCode(qrcodeDiv, {
                text: link,
                width: 150,
                height: 150
            });
        }
        
        function copyShareLink() {
            const input = document.getElementById("share-link-inp");
            input.select();
            document.execCommand("copy");
            alertMsg("Copied", "Link copied!");
        }
        
        function copyReferral() {
            if (!currentUser) return;
            navigator.clipboard.writeText(currentUser.uiId);
            alertMsg("Copied", "ID Copied");
        }
        
        // কমিশন ডিটেইল
        async function loadCommissionDetails() {
            try {
                const response = await fetch('index.php?ajax=getCommissionDetails');
                const result = await response.json();
                
                if (result.success) {
                    let html = '';
                    result.commissions.forEach(comm => {
                        html += `<tr>
                            <td>${new Date(comm.created_at).toLocaleDateString()}</td>
                            <td>${comm.type}</td>
                            <td class="text-green">+${parseFloat(comm.amount).toFixed(2)}</td>
                            <td>${comm.from_uiId || ''}</td>
                        </tr>`;
                    });
                    
                    if (document.getElementById('comm-detail-body')) {
                        document.getElementById('comm-detail-body').innerHTML = html;
                    }
                }
            } catch (error) {
                console.error('Error loading commission details:', error);
            }
        }
        
        // ইউআই আপডেট
        function updateUI() {
            if (!currentUser) return;
            
            const bal = parseFloat(currentUser.depositBalance) + parseFloat(currentUser.winningBalance);
            
            // হেডার ব্যালেন্স
            const headerBalance = document.getElementById('header-balance');
            if (headerBalance) headerBalance.innerText = bal.toFixed(2);
            
            // ওয়ালেট ব্যালেন্স
            const walletTotal = document.getElementById('wallet-total-big');
            if (walletTotal) walletTotal.innerText = "৳" + bal.toFixed(2);
            
            const walletTurnover = document.getElementById('wallet-turnover');
            if (walletTurnover) walletTurnover.innerText = currentUser.turnoverReq || '0.00';
            
            const walletDep = document.getElementById('wallet-dep');
            if (walletDep) walletDep.innerText = "৳" + (currentUser.depositBalance || '0.00');
            
            const walletWin = document.getElementById('wallet-win');
            if (walletWin) walletWin.innerText = "৳" + (currentUser.winningBalance || '0.00');
            
            const wAvailBal = document.getElementById('w-avail-bal');
            if (wAvailBal) wAvailBal.innerText = "৳" + (currentUser.winningBalance || '0.00');
            
            const betBal = document.getElementById('bet-bal');
            if (betBal) betBal.innerText = bal.toFixed(2);
            
            // প্রোমোশন
            const promoTotalComm = document.getElementById('promo-total-comm');
            if (promoTotalComm) promoTotalComm.innerText = currentUser.totalCommission || '0.00';
            
            const promoDirectReg = document.getElementById('promo-direct-reg');
            if (promoDirectReg) promoDirectReg.innerText = currentUser.directSubs || 0;
            
            const promoCodeDisp = document.getElementById('promo-code-disp');
            if (promoCodeDisp) promoCodeDisp.innerText = currentUser.uiId;
        }
        
        // পার্টনার গেম জেনারেট
        function generatePartnerGames() {
            const container = document.getElementById('partner-grid-container');
            if (!container) return;
            
            let html = '';
            for (let i = 1; i <= 50; i++) {
                let imgNum = (i % 9) + 1;
                html += `
                <a href="https://google.com" target="_blank" class="partner-game-card">
                    <img src="ball_${imgNum}.webp" alt="Game ${i}">
                    <div class="pg-overlay">Game ${i}</div>
                </a>`;
            }
            container.innerHTML = html;
        }
        
        // অ্যালার্ট ফাংশন
        function alertMsg(t, m) {
            document.getElementById('alert-t').innerText = t;
            document.getElementById('alert-m').innerText = m;
            document.getElementById('alert-overlay').style.display = 'flex';
        }
        
        function closeResPopup() {
            document.getElementById('res-overlay').style.display = 'none';
        }
        
        // লগআউট
        function logout() {
            window.location.href = 'index.php?logout=1';
        }
        
        // পেজ লোড হলে
        document.addEventListener('DOMContentLoaded', function() {
            // পার্টনার গেম জেনারেট
            generatePartnerGames();
            
            // ডিপোজিট মেথড সেট
            loadAgentNum();
            
            // URL থেকে গেম টাইপ পেতে
            const urlParams = new URLSearchParams(window.location.search);
            const page = urlParams.get('page');
            const gameType = urlParams.get('type');
            
            if (page === 'game' && gameType && GAMES[gameType]) {
                activeGame = gameType;
                setTimeout(runTimer, 500);
                setTimeout(loadGameHistory, 1000);
            }
            
            // কমিশন ডিটেইল লোড (যদি কমিশন পেজ হয়)
            if (page === 'commission-detail') {
                loadCommissionDetails();
            }
            
            // অ্যালার্ট দেখান (যদি থাকে)
            <?php if (isset($_SESSION['alert'])): ?>
            alertMsg('<?php echo $_SESSION['alert']['type']; ?>', '<?php echo addslashes($_SESSION['alert']['message']); ?>');
            <?php 
            unset($_SESSION['alert']); 
            endif; ?>
        });
    </script>
</body>
</html>